package com.yee.product.util;

import java.beans.BeanInfo;
import java.beans.Introspector;
import java.beans.PropertyDescriptor;
import java.lang.reflect.Method;
import java.util.ArrayList;
import java.util.Collections;
import java.util.HashMap;
import java.util.Iterator;
import java.util.Map;

import org.apache.commons.codec.digest.DigestUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.util.Assert;
import org.springframework.util.ObjectUtils;
import org.springframework.util.StringUtils;

public class SignUtil {

	private static final Logger logger = LoggerFactory.getLogger(SignUtil.class);

	/**
	 * 参数md5签名
	 * @param map 待签名map<String,String[]>
	 * @param key 签名key
     * @param ignores 要忽略的参数
	 * @return 返回MD5签名值
	 */
	public static String md5SignArray(Map<String, String[]> map, String key, String... ignores) {
		Assert.notEmpty(map, "参数不能为空");
		Assert.hasText(key, "密钥不能为空");
		//如果有要忽略参数
		if (!ObjectUtils.isEmpty(ignores)) {
		    Assert.isTrue(map.size() > ignores.length, "忽略参数数量>=对象参数");
            //清除map中要忽略的参数
            for (String ig: ignores) {
                map.remove(ig);
            }
		}

		ArrayList<String> paramNames = new ArrayList<String>(map.keySet());
		Collections.sort(paramNames);
		StringBuilder signSource = new StringBuilder();
		Iterator<String> iterator = paramNames.iterator();
		while (iterator.hasNext()) {
			String paramName = iterator.next();
			String[] paramValues = map.get(paramName);
			if (ObjectUtils.isEmpty(paramValues)) {
				continue;
			}
			for (String paramValue : paramValues) {
				if (StringUtils.hasText(paramValue)) {
					//key=value&
					signSource.append(paramName).append("=").append(paramValue).append("&");
				}
			}
		}
		if (signSource.length() > 0) {
			//去除最后一个&
			signSource.deleteCharAt(signSource.length() - 1);
		}

		signSource.append(key);
		return DigestUtils.md5Hex(signSource.toString()).toUpperCase();
	}

    /**
     * 参数md5签名
     * @param map 待签名map<String,String[]>
     * @param key 签名key
     * @return 返回MD5签名值
     */
    public static String md5SignArray(Map<String, String[]> map, String key) {
        return md5SignArray(map, key);
    }

	/**
	 * 参数md5签名
	 * @param map 待签名map<String,String>
	 * @param key 签名key
     * @param ignores 要忽略的参数
	 * @return 返回MD5签名值
	 */
	public static String md5Sign(Map<String, String> map, String key, String... ignores) {
		Assert.notEmpty(map, "参数不能为空");
		Assert.hasText(key, "密钥不能为空");

        //如果有要忽略参数
        if (!ObjectUtils.isEmpty(ignores)) {
            Assert.isTrue(map.size() > ignores.length, "忽略参数个数>=对象参数");
            //清除map中要忽略的参数
            for (String ig: ignores) {
                map.remove(ig);
            }
        }

		ArrayList<String> paramNames = new ArrayList<String>(map.keySet());
		Collections.sort(paramNames);
		StringBuilder signSource = new StringBuilder();
		Iterator<String> iterator = paramNames.iterator();
		while (iterator.hasNext()) {
			String paramName = iterator.next();
			String paramValue = map.get(paramName);
			if (!StringUtils.hasText(paramValue)) {
				continue;
			}
			//key=value&
			signSource.append(paramName).append("=").append(paramValue).append("&");
		}
		if (signSource.length() > 0) {
			//去除最后一个&
			signSource.deleteCharAt(signSource.length() - 1);
		}

		signSource.append(key);
		logger.info(signSource.toString());
		return DigestUtils.md5Hex(signSource.toString()).toUpperCase();
	}

	/**
	 * 参数md5签名
	 * @param object 待签名对象，空属性不参与签名
	 * @param key 签名key
     * @param ignores 要忽略的参数
	 * @return 返回MD5签名值
	 */
	public static String md5Sign(Object object, String key, String... ignores) {
		Assert.notNull(object, "object not emtpy");
		Map<String, String> map = new HashMap<String, String>();
		try {
			BeanInfo beanInfo = Introspector.getBeanInfo(object.getClass(), Object.class);
			PropertyDescriptor[] propertyDescriptors = beanInfo.getPropertyDescriptors();
			for (PropertyDescriptor property : propertyDescriptors) {
				String paramName = property.getName();

				Method getter = property.getReadMethod();
				Object value = getter != null ? getter.invoke(object) : null;
				String valueStr = value == null ? "" : value.toString();
				map.put(paramName, valueStr);
			}
			return md5Sign(map, key, ignores);
		} catch (Exception e) {
			logger.error("md5Sign by object is error", e);
		}
		return null;
	}

}